 /***************************************************************************
 *   Copyright (C) 2006 by Emanuel Wegh                                    *
 *   maan@ddsw.nl                                                          *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <QtCore>

#include "cameracontrol.h"
#include "image.h"
#include "jpegcompress.h"

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <errno.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <linux/videodev.h>
#include <math.h>

CameraControl::CameraControl(QObject *parent) : QObject(parent)
{
	// init
	cameraImage = new QByteArray();
	image = new Image(VIDEO_WIDTH, VIDEO_HEIGHT);
	jpegImage = new JpegCompress(VIDEO_WIDTH, VIDEO_HEIGHT);
}

CameraControl::~CameraControl()
{
	close(videoFD);
	
	delete(image);
	delete(jpegImage);
	delete(cameraImage);
}

bool CameraControl::initCamera()
{
	video_capability cameraCapability;

	// Open video device
	videoFD = open(VIDEO_DEVICE, O_RDWR);
	if (videoFD == -1)
	{
		perror("Camera");
		return false;
	}

	// Retreive camera capability
	if (ioctl(videoFD, VIDIOCGCAP, &cameraCapability) == -1)
	{
		perror("Camera capability");
		return false;
	}
	else
		printf("Connected camera: %s\n", cameraCapability.name);

	// Set video window to correct size
	ioctl(videoFD, VIDIOCGWIN, &videoWindow);			// get default values

	videoWindow.x = 0;
	videoWindow.y = 0;
	videoWindow.width = VIDEO_WIDTH;
	videoWindow.height = VIDEO_HEIGHT;
	videoWindow.clipcount = 0;
	if (ioctl(videoFD, VIDIOCSWIN, &videoWindow) == -1)	// set new values
	{
		perror("Video window");
		return false;
	}
    
	// Retreive image properties    
	ioctl(videoFD, VIDIOCGPICT, &imageProperties);		// get default values
	ioctl(videoFD, VIDIOCGPICT, &lowPowerImageProperties);	// get default values

	// Set image properties to correct values
	imageProperties.brightness = 50000;
 	imageProperties.contrast = 10000;
	imageProperties.hue = 0;							// no function
	imageProperties.colour = 0;							// no function
	
	// Set low power values
	lowPowerImageProperties.brightness = 50000;
 	lowPowerImageProperties.contrast = 60000;			// this option costs less power !!!!
	lowPowerImageProperties.hue = 0;					// no function
	lowPowerImageProperties.colour = 0;					// no function	
	
	if (ioctl(videoFD, VIDIOCSPICT, &lowPowerImageProperties) == -1)	// set new values
	{
		perror("Image properties");
		return false;
	}


	// Camera succesfuly initialized
	return true;
}

bool CameraControl::readImage()
{
	// Used when measuring camera times.
	//QTime t;
	//t.start();	// timer
	
	// Read image from camera
	if (read(videoFD, image->imageBuffer, image->imageSize) != image->imageSize)
	{
		perror("Reading image");
		return false;
	}
	
	//printf("Image time 1: %d\n", t.elapsed());
	
	// Swap red and blue channel. Who swapped it before????
	image->swapRedBlueChannel();

	// Blue channel is to weak
	image->changeBlueIntensity(130);
	
	//printf("Image time 2: %d\n", t.elapsed());

	// Set camera integration time for next image
	imageProperties.contrast = calcIntegrationTime();
	//printf("New integration time: %d\n", imageProperties.contrast);
	if (ioctl(videoFD, VIDIOCSPICT, &imageProperties) == -1)	// set new values
	{
		perror("Image properties");
		return false;
	}
	
	//printf("Image time 3: %d\n", t.elapsed());
	
	// Jpeg compression (write file to /tmp)
	jpegImage->writeJpeg(image->imageBuffer);
	
	//printf("Image time 4: %d\n", t.elapsed());
	
	// Read Jpeg in QByteArray
	QFile jpegImage("/tmp/webcam.jpeg", this);
	jpegImage.open(QIODevice::ReadOnly);
	cameraImage->clear();
	*cameraImage = jpegImage.read(jpegImage.size());
	jpegImage.close();
	
	// Reading image successful
	return true;
}

int CameraControl::calcIntegrationTime()
{
	// Calculate new integration time for next exposure

	int currentIntegrationTime = imageProperties.contrast, newIntegrationTime;
	float factor = 0;

	/*
	The used factor is a power of 2.
	The same factor is influenced by 2 functions, which work in opposite direction.
	A factor of +1.0 multiplies the integration time by 2.
	A factor of  0.0 does nothing.
	A factor of -1.0 divides the integration time by 2.
	An image, which is overexposed, has much more high exposed pixels than low exposed ones.
	This will drop the factor below zero, which reduces the next integration time.
	This procedure must be repeated a few times before the right integration time is acquired.
	*/ 

	factor -= 1.0 * image->highExposedPixels();
	//printf("factor: %f ", factor);
	factor += 1.0 * image->lowExposedPixels();
	//printf("factor: %f ", factor);   

	newIntegrationTime = (int)(currentIntegrationTime * powf(2, factor));
    
	// Check on under/overflow conditions
	if (newIntegrationTime > 65535)
		newIntegrationTime = 65535;
	if (newIntegrationTime < 0)
		newIntegrationTime = 0;
		
	// Dont't use little integration changes, to reduce flickering
	if ((float)abs(newIntegrationTime - currentIntegrationTime) / currentIntegrationTime < .05)
		newIntegrationTime = currentIntegrationTime;

	return newIntegrationTime;
}


// A high 'contrast' camera setting gives low CPU usage (si = software int. in kernel).
// This results in low power usage.
void CameraControl::setLowPowerMode()
{
	if (ioctl(videoFD, VIDIOCSPICT, &lowPowerImageProperties) == -1)	// set new values
		perror("Image properties");
}

// Restore last used camera settings.
void CameraControl::restoreNormalPowerMode()
{
	if (ioctl(videoFD, VIDIOCSPICT, &imageProperties) == -1)			// set new values
		perror("Image properties");
}
