/***************************************************************************
 *   Copyright (C) 2006 by Emanuel Wegh                                    *
 *   maan@ddsw.nl                                                          *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <QtCore>
#include <QtNetwork>

#include "robotserver.h"
#include "robotcontrol.h"
#include "cameracontrol.h"

#include "stdlib.h"

RobotServer::RobotServer(QObject *parent) : QTcpServer(parent)
{
	// init
	commSelector = NO_SELECTOR;
	robotClientConnected = false;
	tcpClientConnected = false;
	
	robotControl = new RobotControl(this);
	cameraControl = new CameraControl(this);

	// heartbeat
	heartBeat = new QTimer(this);
	heartBeatSeen = false;
	connect(heartBeat, SIGNAL(timeout()), this, SLOT(checkHeartBeat()));
	
	// camera
	imageProcessing = new QTimer(this);
	connect(imageProcessing, SIGNAL(timeout()), this, SLOT(sendImage()));
	
	// Auto drive timing
	autoDriveTimer = new QTimer(this);
	connect(autoDriveTimer, SIGNAL(timeout()), this, SLOT(autoDriveDisconnectPower()));
	
	// auto control test system
	autoControlTimer = new QTimer(this);
	connect(autoControlTimer, SIGNAL(timeout()), this, SLOT(autoControl()));
}

bool RobotServer::listen()
{        
	// Init robot: TCP port, Teleo system and camera.

	QTextStream consoleText(stderr);

	// Start tcp server on port TCP_PORT
	if (!QTcpServer::listen(QHostAddress::Any, TCP_PORT))
	{
		consoleText << "Can't use port ";
		consoleText << TCP_PORT;
		consoleText << "\n";
		return false;
	}

	// Init Teleo system
	if (!robotControl->startTeleo())
	{
		consoleText << "Teleo system isn't functioning\n";
		return false;
	}
	else
		consoleText << "Teleo system initialized\n";
	
	// Init camera
	if (!cameraControl->initCamera())
	{
		consoleText << "Camera isn't functioning\n";
		return false;
	}

	consoleText << "Robot server started\n";

	connect(this, SIGNAL(newConnection()), this, SLOT(startCommunication()));
	
	// Start auto control test
	// autoControlTimer->start(1000);

	// Robot server initialized correctly
	return true;
}

void RobotServer::startCommunication()
{
	// Start new communication session with robot client.

	QByteArray block;
	QDataStream buffer(&block, QIODevice::ReadWrite);
	
	if(!tcpClientConnected)
	{	
		// Start robot connection (there are no other connections).
		tcpClientConnected = true;
		
		// Start new socket
		clientConnection = new QTcpSocket(this);
		clientConnection = nextPendingConnection();
	
		// Delete connection and object after client disconnects.
		connect(clientConnection, SIGNAL(disconnected()), clientConnection, SLOT(deleteLater()));
		
		connect(clientConnection, SIGNAL(disconnected()), this, SLOT(endCommunication()));
		// Start robot control
		connect(clientConnection, SIGNAL(readyRead()), this, SLOT(readClientData()));

		// Turn on connection LED
		robotControl->turnOnConnectLed();

		// First send server verion information
		buffer << ROBOT_VERSION_INFO;	// communication selector
		buffer << ROBOT_VERSION;
		
		clientConnection->write(block);
	}
	else
	{	
		// Block robot connection (there is already another connection).
		// Open connection only temporary to send information to client.
		
		// Start new temp. socket
		QTcpSocket *tempClientConnection = new QTcpSocket(this);
		tempClientConnection = nextPendingConnection();
	
		// Delete connection and object after client disconnects.
		connect(tempClientConnection, SIGNAL(disconnected()), tempClientConnection, SLOT(deleteLater()));

		// Write to console
		printf("A new connection was blocked\n");
						
		// Send 'already connected' signal to client
		buffer << ALREADY_CONNECTED;
		
		tempClientConnection->write(block);
		tempClientConnection->disconnectFromHost();
	}
}

void RobotServer::readClientData()
{
	QDataStream buffer(clientConnection);
	QTextStream consoleText(stdout);
	//QString str;
	int speed, steering, commIdentifier;

	while(clientConnection->bytesAvailable() > 0)
	{
		switch(commSelector)
		{
			// Read communication selector
			case NO_SELECTOR:
				if (clientConnection->bytesAvailable() < sizeof(qint8))
					return;
				else
					buffer >> commSelector;
	
				break;

			// Read robot client version
			case ROBOT_VERSION_INFO:
				if (clientConnection->bytesAvailable() < sizeof(float))
					return;
				else
				{
					buffer >> robotClientVersion;
					commSelector = NO_SELECTOR;		// reset commSelector   
				}
	
				break;
		
			// Read communication ID, needed to accept a connection
			case COMM_ID_INFO:
				if (clientConnection->bytesAvailable() < sizeof(int))
					return;
				else
				{
					buffer >> commIdentifier;
			
					// Check if communication ID is correct
					if (commIdentifier == COMM_ID)
					{
						consoleText << "Robot client connected: "
									<< clientConnection->peerAddress().toString()
									<< "\n";
						robotClientConnected = true;

						//   ----  Communication session is now initialized  -----
			
						// Start checking heartbeat
						heartBeat->start(HEART_BEAT_PERIOD);
						
						// Camera in normal operation
						cameraControl->restoreNormalPowerMode();
						
						// Start sending images
						imageProcessing->start(IMAGE_SEND_PERIOD);
						
						// Drive robot until external power is disconnected
						// Switch to battery power
						robotControl->switchToAccuPower();
						
						// Drive until disconnected
						robotControl->setSpeed(MAX_SPEED);
						autoDriveTimer->start(100);
					}
					else
					{
						consoleText << "Connection refused, no robot client\n";
						clientConnection->disconnectFromHost();
					}

					commSelector = NO_SELECTOR;		// reset commSelector
				}
		
				break;

			// Read heartbeat
			case HEART_BEAT:
				heartBeatSeen = true;
				commSelector = NO_SELECTOR;		// reset commSelector
		
				break;

			// Load image from camera and send to client
			case SELECT_IMAGE:	
				sendImage();
	
				commSelector = NO_SELECTOR;		// reset commSelector
		
				break;

			// Read speed
    		// Only use this information with an accepted connection, for security.
			case SELECT_SPEED:
				if (clientConnection->bytesAvailable() < sizeof(int))
					return;
				else
				{
					buffer >> speed;
					//str.setNum(speed);
					//consoleText << str << " ";
					if (robotClientConnected)
						robotControl->setSpeed(speed);
	
					commSelector = NO_SELECTOR;		// reset commSelector
				}
	
				break;
		
			// Read steering
			// Only use this information with an accepted connection, for security.
			case SELECT_STEERING:
				if (clientConnection->bytesAvailable() < sizeof(int))
					return;
				else
				{
					buffer >> steering;
					if (robotClientConnected)
						robotControl->setSteering(steering);

					commSelector = NO_SELECTOR;		// reset commSelector
				}
	
				break;	
		}
	}
}

void RobotServer::endCommunication()
{
	// When client disconnects: stop the robot!!!

	QTextStream consoleText(stdout);   
	if (robotClientConnected)
		consoleText << "Robot client disconnected\n";

	robotControl->stopMotors();		// fast stop
	robotControl->turnOffConnectLed();
	
	// Put camera in low power mode
	cameraControl->setLowPowerMode();
	
	// Switch to external power if available
	robotControl->switchToExtPower();

	robotClientConnected = false;
	tcpClientConnected = false;
	imageProcessing->stop();
	heartBeat->stop();
}

void RobotServer::checkHeartBeat()
{
	// Check heartbeat. Connection is bad when not seen in time.
	// Stop the robot!!!

	if (heartBeatSeen)
		// Communication OK. Heartbeat seen.
		heartBeatSeen = false;
	else
	{
		// Communication not OK, to slow. Heartbeat not seen.
		QTextStream consoleText(stdout);   
		consoleText << "Bad connection\n";

		robotControl->stopMotors();		// fast stop
	}
}

void RobotServer::sendImage()
{
	QDataStream buffer(clientConnection);
	
	//QTime t;
	//t.start();	// timer
	
	// Read image from camera
	if(!cameraControl->readImage())	// iets met events, minder blokkeren eventloop? !!!!!!!!!!!!!
		printf("readImage: no\n");
	
	buffer << SELECT_IMAGE;									// communication selector
	buffer << (qint32)cameraControl->cameraImage->size();	// write size
	//printf("Compressed size: %d\n", cameraControl->cameraImage->size());
	clientConnection->write(*cameraControl->cameraImage);	// send image
	
	//printf("Total image time: %d\n", t.elapsed());
}


void RobotServer::stopRobotControl()
{
	// Stop all robot activity when quiting this application.
	delete(robotControl);
}


void RobotServer::autoDriveDisconnectPower()
{
	static int i = 0;
	
	if (!robotControl->getExtPowerStatus())
	{
		autoDriveTimer->stop();
		robotControl->setSpeed(0);
		
		printf("debug opletten: %d", i);	// debug: soms stoppen de motoren direkt na een client connect
		// daarna ook geen switch meer naar ext power, terwijl deze wel verbonden is. Het lijkt alsof i klein is.
		// was laatst: 0
		// misschien te snel client connect na starten server: waarschijnlijk niet.
		// ook na starten van server (en clientconnect) en tegelijkertijd zonder melding 'On ext'.
		// dit betekend dat extPowerStatus false is terwijl deze eigenlijk true moet zijn.
	}
	else
		i++;
		
	if (i * autoDriveTimer->interval() >= 6000)
	{
		autoDriveTimer->stop();
		robotControl->setSpeed(0);
		i = 0;
		//robotControl->switchToExtPower();
	}
}


// Auto control test system.
// Robot changes speed automatically for testing purpose.
void RobotServer::autoControl()
{
	static int speed = 0, interval = 5;
		
	if(speed > 70)
		interval = -5;
	if(speed < -70)
		interval = 5;
		
	speed += interval;
	
	
	//if(speed <= 0)
	{
		robotControl->setSteering(0);
		robotControl->setSpeed(speed);
	}
}
